/*
*  Object %name    : %
*  State           :  %state%
*  Creation date   :  Mar 29 2007
*  Last modified   :  %modify_time%
*/
/** @file
*  \brief A brief description of this module
*
*  \version CE2_TST_CMLA.h#1:incl:1
*  \author Yermalayeu Ihar
*  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
*           All Rights reserved
*/

/************* Include Files ****************/

#include <memory.h>
#include "CryptoEngine2.h"
#include "CE2_TST_Common.h"
#include "CE2_TST_CMLA.h"

/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/
/*********************** Private Function ***********************/
/************************ Public Functions **********************/

/*****************************************************************************
* Function Name:                                                           
*  CE2_TST_CMLA_Wrap_Unwrap		
* 
* Inputs:
*  @param[in/out] out - output stream for logs;
*
* Outputs:
*  @return - On success returns CE2_TST_VALID; else returns CE2_TST_FAILED.
*
* Description:																	
* 	Verify functionality of CE2_TST_CMLA_Wrap and CE2_TST_CMLA_Unwrap functions.
*
* Algorithm:		
*  -#  											
*
******************************************************************************/	
DxStatus CE2_TST_CMLA_Wrap_Unwrap(FILE *out)
{
  DxStatus result = CE2_OK, error;
  CE2_CMLA_KEK_t KEK = {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
  };
  CE2_CMLA_UNWRAPPED_KEY_t  Key1 = {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f
  }, Key2;
  CE2_CMLA_WRAPPED_KEY_t WrapKey;

  PrintTestBeginMsg(out, __FUNCTION__);

  error = CE2_CMLA_Wrap(KEK, Key1, WrapKey);
  if (error != CE2_OK) {
    fprintf(out, "Can't make CE2_CMLA_Wrap!\n");
    result |= CE2_TST_FAILED;
    goto end_test;
  }

  error = CE2_CMLA_Unwrap(KEK, WrapKey, Key2);
  if (error != CE2_OK) {
    fprintf(out, "Can't make CE2_CMLA_Unwrap!\n");
    result |= CE2_TST_FAILED;
    goto end_test;
  }

  if (memcmp(Key1, Key2, CE2_CMLA_UNWRAPPED_KEY_SIZE_IN_BYTES) != 0) {
    fprintf(out, "Wrong CE2_CMLA_Wrap/CE2_CMLA_Unwrap output!\n");
    result |= CE2_TST_FAILED;
  }

end_test:
  PrintTestEndMsg(out, __FUNCTION__, result);

  return result;
}

/*****************************************************************************
* Function Name:                                                           
*  CE2_TST_CMLA_RSA_Encrypt_Decrypt			
* 
* Inputs:
*  @param[in/out] out - output stream for logs;
*
* Outputs:
*  @return - On success returns CE2_TST_VALID; else returns CE2_TST_FAILED.
*
* Description:																	
* 	Verify functionality of CE2_CMLA_RSA_Encrypt and CE2_CMLA_RSA_Decrypt functions.
*
* Algorithm:		
*  -#  											
*
******************************************************************************/	
DxStatus CE2_TST_CMLA_RSA_Encrypt_Decrypt(FILE *out)
{
  DxStatus result = CE2_OK, error;
  CE2_RSAUserPubKey_t UserPublKey;
  CE2_RSAUserPrivKey_t UserPrivKey;
  DxUint8_t publExp[] = {0x01,0x00,0x01};
  CE2_CMLA_RSA_MESSAGE_t Message = {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x10, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x18, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x20, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x28, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x30, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x38, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x40, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x48, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x50, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x58, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x60, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x68, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x70, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x78, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
  }, EncrMessage, Message2;

  PrintTestBeginMsg(out, __FUNCTION__);

  error = CE2_RSA_KG_GenerateKeyPair(publExp, 3, 1024, &UserPrivKey, &UserPublKey);
  if (error != CE2_OK) {
    fprintf(out, "Can't generate RSA key Pair!");
    result |= error;
    goto test_end;
  }

  error = CE2_CMLA_RSA_Encrypt(&UserPublKey, Message, EncrMessage);
  if (error != CE2_OK) {
    fprintf(out, "Can't encrypt message!");
    result |= error;
    goto test_end;
  }

  error = CE2_CMLA_RSA_Decrypt(&UserPrivKey, EncrMessage, Message2);
  if (error != CE2_OK) {
    fprintf(out, "Can't decrypt message!");
    result |= error;
    goto test_end;
  }

  if (memcmp(Message, Message2, CE2_CMLA_RSA_MESSAGE_LEN_BYTES) != 0) {
    fprintf(out, "Wrong CE2_CMLA_RSA_Encrypt/CE2_CMLA_RSA_Decrypt output!\n");
    result |= CE2_TST_FAILED;
  }

test_end:
  PrintTestEndMsg(out, __FUNCTION__, result);

  return result;
}

/*****************************************************************************
* Function Name:                                                           
*  CE2_TST_CMLA_B1			
* 
* Inputs:
*  @param[in/out] out - output stream for logs;
*
* Outputs:
*  @return - On success returns CE2_TST_VALID; else returns CE2_TST_FAILED.
*
* Description:																	
* 	Verify functionality of CMLA (test B1 from standard).
*
* Algorithm:		
*  -#  											
*
******************************************************************************/	
DxStatus CE2_TST_CMLA_B1(FILE *out)
{
  DxStatus result = CE2_OK, error;
  CE2_RSAUserPubKey_t UserPublKey;
  CE2_RSAUserPrivKey_t UserPrivKey;
  /* RSA keys */
  DxUint8_t N[] = {
    0xa2, 0xba, 0x40, 0xee, 0x07, 0xe3, 0xb2, 0xbd,
    0x2f, 0x02, 0xce, 0x22, 0x7f, 0x36, 0xa1, 0x95,
    0x02, 0x44, 0x86, 0xe4, 0x9c, 0x19, 0xcb, 0x41,
    0xbb, 0xbd, 0xfb, 0xba, 0x98, 0xb2, 0x2b, 0x0e,
    0x57, 0x7c, 0x2e, 0xea, 0xff, 0xa2, 0x0d, 0x88,
    0x3a, 0x76, 0xe6, 0x5e, 0x39, 0x4c, 0x69, 0xd4, 
    0xb3, 0xc0, 0x5a, 0x1e, 0x8f, 0xad, 0xda, 0x27,
    0xed, 0xb2, 0xa4, 0x2b, 0xc0, 0x00, 0xfe, 0x88,
    0x8b, 0x9b, 0x32, 0xc2, 0x2d, 0x15, 0xad, 0xd0, 
    0xcd, 0x76, 0xb3, 0xe7, 0x93, 0x6e, 0x19, 0x95, 
    0x5b, 0x22, 0x0d, 0xd1, 0x7d, 0x4e, 0xa9, 0x04,
    0xb1, 0xec, 0x10, 0x2b, 0x2e, 0x4d, 0xe7, 0x75,
    0x12, 0x22, 0xaa, 0x99, 0x15, 0x10, 0x24, 0xc7,
    0xcb, 0x41, 0xcc, 0x5e, 0xa2, 0x1d, 0x00, 0xee,
    0xb4, 0x1f, 0x7c, 0x80, 0x08, 0x34, 0xd2, 0xc6,
    0xe0, 0x6b, 0xce, 0x3b, 0xce, 0x7e, 0xa9, 0xa5
  };
  DxUint8_t e[] = {
    0x01,0x00,0x01
  };
  DxUint8_t p[] = {
    0xd1, 0x7f, 0x65, 0x5b, 0xf2, 0x7c, 0x8b, 0x16,
    0xd3, 0x54, 0x62, 0xc9, 0x05, 0xcc, 0x04, 0xa2,
    0x6f, 0x37, 0xe2, 0xa6, 0x7f, 0xa9, 0xc0, 0xce, 
    0x0d, 0xce, 0xd4, 0x72, 0x39, 0x4a, 0x0d, 0xf7,
    0x43, 0xfe, 0x7f, 0x92, 0x9e, 0x37, 0x8e, 0xfd, 
    0xb3, 0x68, 0xed, 0xdf, 0xf4, 0x53, 0xcf, 0x00,
    0x7a, 0xf6, 0xd9, 0x48, 0xe0, 0xad, 0xe7, 0x57,
    0x37, 0x1f, 0x8a, 0x71, 0x1e, 0x27, 0x8f, 0x6b
  };
  DxUint8_t q[] = {
    0xc6, 0xd9, 0x2b, 0x6f, 0xee, 0x74, 0x14, 0xd1,
    0x35, 0x8c, 0xe1, 0x54, 0x6f, 0xb6, 0x29, 0x87,
    0x53, 0x0b, 0x90, 0xbd, 0x15, 0xe0, 0xf1, 0x49,
    0x63, 0xa5, 0xe2, 0x63, 0x5a, 0xdb, 0x69, 0x34,
    0x7e, 0xc0, 0xc0, 0x1b, 0x2a, 0xb1, 0x76, 0x3f,
    0xd8, 0xac, 0x1a, 0x59, 0x2f, 0xb2, 0x27, 0x57, 
    0x46, 0x3a, 0x98, 0x24, 0x25, 0xbb, 0x97, 0xa3,
    0xa4, 0x37, 0xc5, 0xbf, 0x86, 0xd0, 0x3f, 0x2f
  };
  DxUint8_t dP[] = {
    0x9d, 0x0d, 0xbf, 0x83, 0xe5, 0xce, 0x9e, 0x4b,
    0x17, 0x54, 0xdc, 0xd5, 0xcd, 0x05, 0xbc, 0xb7, 
    0xb5, 0x5f, 0x15, 0x08, 0x33, 0x0e, 0xa4, 0x9f,
    0x14, 0xd4, 0xe8, 0x89, 0x55, 0x0f, 0x82, 0x56,
    0xcb, 0x5f, 0x80, 0x6d, 0xff, 0x34, 0xb1, 0x7a,
    0xda, 0x44, 0x20, 0x88, 0x53, 0x57, 0x7d, 0x08,
    0xe4, 0x26, 0x28, 0x90, 0xac, 0xf7, 0x52, 0x46,
    0x1c, 0xea, 0x05, 0x54, 0x76, 0x01, 0xbc, 0x4f
  };
  DxUint8_t dQ[] = {
    0x12, 0x91, 0xa5, 0x24, 0xc6, 0xb7, 0xc0, 0x59,
    0xe9, 0x0e, 0x46, 0xdc, 0x83, 0xb2, 0x17, 0x1e, 
    0xb3, 0xfa, 0x98, 0x81, 0x8f, 0xd1, 0x79, 0xb6, 
    0xc8, 0xbf, 0x6c, 0xec, 0xaa, 0x47, 0x63, 0x03,
    0xab, 0xf2, 0x83, 0xfe, 0x05, 0x76, 0x9c, 0xfc, 
    0x49, 0x57, 0x88, 0xfe, 0x5b, 0x1d, 0xdf, 0xde, 
    0x9e, 0x88, 0x4a, 0x3c, 0xd5, 0xe9, 0x36, 0xb7, 
    0xe9, 0x55, 0xeb, 0xf9, 0x7e, 0xb5, 0x63, 0xb1
  };
  DxUint8_t qInv[] = {
    0xa6, 0x3f, 0x1d, 0xa3, 0x8b, 0x95, 0x0c, 0x9a, 
    0xd1, 0xc6, 0x7c, 0xe0, 0xd6, 0x77, 0xec, 0x29,
    0x14, 0xcd, 0x7d, 0x40, 0x06, 0x2d, 0xf4, 0x2a, 
    0x67, 0xeb, 0x19, 0x8a, 0x17, 0x6f, 0x97, 0x42, 
    0xaa, 0xc7, 0xc5, 0xfe, 0xa1, 0x4f, 0x22, 0x97, 
    0x66, 0x2b, 0x84, 0x81, 0x2c, 0x4d, 0xef, 0xc4, 
    0x9a, 0x80, 0x25, 0xab, 0x43, 0x82, 0x28, 0x6b,
    0xe4, 0xc0, 0x37, 0x88, 0xdd, 0x01, 0xd6, 0x9f
  };
  DxUint8_t PlanText[] = {
    0x7c, 0x09, 0xa0, 0x0a, 0xc8, 0x05, 0x88, 0x6e, 
    0x80, 0x42, 0x7a, 0xf1, 0x0f, 0xe9, 0xef, 0x21, 
    0xd9, 0x74, 0x4c, 0xc4, 0x85, 0x51, 0x90, 0x3c, 
    0xec, 0xa8, 0xd9, 0x24, 0x45, 0xff, 0x7d, 0x6c, 
    0x31, 0x04, 0x67, 0xdc, 0x31, 0xaa, 0xa5, 0x68, 
    0xfe, 0x45, 0xe9, 0xcb, 0x47, 0xe7, 0x36, 0xfb, 
    0x53, 0x5f, 0xef, 0x87, 0x81, 0xe2, 0xd2, 0xfd, 
    0x09, 0x13, 0xd8, 0xe1, 0x03, 0xd5, 0xac, 0x82, 
    0x18, 0x6b, 0x16, 0x2b, 0x9c, 0x25, 0xff, 0x38, 
    0x50, 0xd6, 0x90, 0x8a, 0x9c, 0x78, 0xc9, 0x88, 
    0xd3, 0x59, 0xad, 0x5d, 0x7b, 0xf7, 0x66, 0x8f, 
    0xde, 0x64, 0x20, 0xa8, 0x35, 0x5f, 0x45, 0x6d, 
    0xb1, 0xc6, 0xb2, 0x9c, 0x1a, 0x18, 0x94, 0xd3, 
    0x83, 0x2f, 0x21, 0x86, 0x0a, 0x92, 0xeb, 0x9d, 
    0x3e, 0x69, 0x6c, 0x2e, 0x59, 0x24, 0xf9, 0x23, 
    0xd1, 0x32, 0xd2, 0xf1, 0x73, 0x3b, 0xff, 0xb4
  };
  DxUint8_t CryptedText[] = {
    0x83, 0xd7, 0xe0, 0xe0, 0xfd, 0x69, 0xc7, 0xd9, 
    0xdf, 0xc7, 0x83, 0x84, 0x88, 0xce, 0xf1, 0xab, 
    0xb7, 0x96, 0xca, 0x84, 0x65, 0x14, 0x01, 0x72, 
    0x16, 0x37, 0x41, 0x90, 0xc5, 0x2d, 0x88, 0x17, 
    0xdd, 0x85, 0x39, 0x38, 0xef, 0x15, 0x28, 0x5f, 
    0x7a, 0x0d, 0xdc, 0xe5, 0xed, 0xc7, 0x4d, 0x10, 
    0x66, 0x61, 0xe7, 0xe9, 0xf1, 0xd7, 0xc4, 0x7c, 
    0x21, 0x02, 0xc4, 0x29, 0x45, 0xbb, 0x8f, 0x07, 
    0xda, 0x57, 0x65, 0x21, 0xda, 0x83, 0xa3, 0xe6, 
    0xa3, 0xf3, 0x92, 0xca, 0x9d, 0xeb, 0xe3, 0xe1, 
    0xcd, 0x1d, 0xf3, 0xe1, 0x05, 0x4c, 0x64, 0xff, 
    0x4c, 0x93, 0x18, 0xdc, 0x56, 0x1c, 0xb5, 0x4e, 
    0xe8, 0xd2, 0x5a, 0x5f, 0x41, 0x33, 0xd8, 0x07, 
    0x81, 0x0a, 0x82, 0x4c, 0xca, 0x5f, 0x7d, 0xaa, 
    0xca, 0x9b, 0xcb, 0x9e, 0x83, 0x3f, 0xaa, 0x79, 
    0xc8, 0xee, 0xc6, 0x9d, 0xb1, 0x19, 0x22, 0xc1
  };
  DxUint8_t buffer0[128], buffer1[128];
  CE2_CMLA_KEK_t KEK = {
    0x74, 0x98, 0x52, 0xe9, 0xdd, 0x93, 0x9a, 0x34, 
    0x5c, 0x80, 0x57, 0x81, 0xf7, 0x2c, 0x7e, 0xde
  }, KEK0;
  CE2_CMLA_UNWRAPPED_KEY_t  Key = {
    0x1d, 0x55, 0xbd, 0x0c, 0x59, 0xd7, 0x90, 0x0c, 
    0x9c, 0x16, 0x3a, 0x15, 0x0d, 0xd3, 0x09, 0xf7, 
    0x95, 0x27, 0x57, 0x09, 0x93, 0xc7, 0xd9, 0x77, 
    0x02, 0x8d, 0xae, 0xb2, 0x2c, 0x34, 0x5c, 0x66
  }, Key0;
  CE2_CMLA_WRAPPED_KEY_t WrapKey = {
    0x96, 0x5d, 0xe9, 0x95, 0x93, 0x61, 0xc2, 0x2f, 
    0xd9, 0x93, 0xd8, 0x48, 0x22, 0xf4, 0x27, 0x1a, 
    0x41, 0x3c, 0x87, 0x99, 0x84, 0x7f, 0x75, 0x20, 
    0x49, 0xe2, 0x8d, 0xf4, 0xa0, 0x71, 0xce, 0xda, 
    0xa8, 0x64, 0x0d, 0x9e, 0x12, 0x3f, 0x7f, 0xa4
  }, WrapKey0;


  PrintTestBeginMsg(out, __FUNCTION__);

  /* CE2_CMLA_RSA_Encrypt/CE2_CMLA_RSA_Decrypt test */
  error = CE2_RSA_Build_PubKey(&UserPublKey, e, sizeof(e), N, sizeof(N));
  if (error != CE2_OK) {
    fprintf(out, "Can't build public key!");
    result |= error;
    goto test_end;
  }

  error = CE2_RSA_Build_PrivKeyCRT(&UserPrivKey, p, sizeof(p), q, sizeof(q),
    dP, sizeof(dP), dQ, sizeof(dQ), qInv, sizeof(qInv));
  if (error != CE2_OK) {
    fprintf(out, "Can't build private key!");
    result |= error;
    goto test_end;
  }

  error = CE2_CMLA_RSA_Encrypt(&UserPublKey, PlanText, buffer0);
  if (error != CE2_OK) {
    fprintf(out, "Can't encrypt message!");
    result |= error;
    goto test_end;
  }

  if (memcmp(buffer0, CryptedText, CE2_CMLA_RSA_MESSAGE_LEN_BYTES) != 0) {
    fprintf(out, "Wrong CE2_CMLA_RSA_Encrypt output!\n");
    result |= CE2_TST_FAILED;
    goto test_end;
  }

  error = CE2_CMLA_RSA_Decrypt(&UserPrivKey, buffer0, buffer1);
  if (error != CE2_OK) {
    fprintf(out, "Can't decrypt message!");
    result |= error;
    goto test_end;
  }

  if (memcmp(buffer1, PlanText, CE2_CMLA_RSA_MESSAGE_LEN_BYTES) != 0) {
    fprintf(out, "Wrong CE2_CMLA_RSA_Decrypt output!\n");
    result |= CE2_TST_FAILED;
    goto test_end;
  }

  /* CE2_CMLA_KDF test */
  error = CE2_CMLA_KDF(PlanText, KEK0);
  if (error != CE2_OK) {
    fprintf(out, "Can't make CMLA KDF!");
    result |= error;
    goto test_end;
  }

  if (memcmp(KEK0, KEK, CE2_CMLA_KEK_SIZE_IN_BYTES) != 0) {
    fprintf(out, "Wrong CE2_CMLA_KDF output!\n");
    result |= CE2_TST_FAILED;
    goto test_end;
  }

  /* CE2_CMLA_Wrap/CE2_CMLA_Unwrap test */
  error = CE2_CMLA_Wrap(KEK, Key, WrapKey0);
  if (error != CE2_OK) {
    fprintf(out, "Can't make CMLA Wrap!");
    result |= error;
    goto test_end;
  }

  if (memcmp(WrapKey0, WrapKey, CE2_CMLA_WRAPPED_KEY_SIZE_IN_BYTES) != 0) {
    fprintf(out, "Wrong CE2_CMLA_Wrap output!\n");
    result |= CE2_TST_FAILED;
    goto test_end;
  }

  error = CE2_CMLA_Unwrap(KEK, WrapKey0, Key0);
  if (error != CE2_OK) {
    fprintf(out, "Can't make CMLA Unwrap!");
    result |= error;
    goto test_end;
  }

  if (memcmp(Key0, Key, CE2_CMLA_UNWRAPPED_KEY_SIZE_IN_BYTES) != 0) {
    fprintf(out, "Wrong CE2_CMLA_Unwrap output!\n");
    result |= CE2_TST_FAILED;
    goto test_end;
  }


test_end:
  PrintTestEndMsg(out, __FUNCTION__, result);

  return result;
}
